#!/usr/bin/env python
"""
Syntax: MergeActor.py ticketnum component version requestor

Merge a branch to its trunk.
"""

import os
import shutil

from mergebot import SvnOps
from mergebot.Actor import Actor

class MergeActor(Actor):
    """Merges a branch to the line of development on which it is based.
    """
    def execute(self):
        "Merge a branch to its trunk"
        results = {}
        logfile = self.logfilename()
        checkoutdir = self.work_dir
        # Delete the working directory so we get a completely clean working
        # copy.
        if os.path.exists(checkoutdir):
            shutil.rmtree(checkoutdir)

        # Make sure the various urls we require do exist
        if not SvnOps.get_branch_info(self.local_url(), logfile):
            comment = 'Component %s does not exist in the repository.' \
                % self.component
            return results, comment, False
        if not SvnOps.get_branch_info(self.local_url() + '/branches', logfile):
            comment = 'No directory in which to create branches for component %s in the repository.' % self.component
            return results, comment, False
        if not SvnOps.get_branch_info(self.baseline_local_url(), logfile):
            comment = 'Version %s for component %s does not exist in the repository.' % (self.version, self.component)
            return results, comment, False

        rev_info = SvnOps.get_branch_info(self.branch_local_url(), logfile)
        if not rev_info:
            comment = 'Branch for ticket %s does not exist in the repository.' % (self.ticket)
            return results, comment, False
        startrev, endrev = rev_info

        SvnOps.checkout(self.baseline_local_url(), checkoutdir, logfile)
        # FIXME: check return code
        merge_results = SvnOps.merge(self.branch_local_url(), checkoutdir,
                               (startrev, endrev), logfile)
        conflicts = SvnOps.conflicts_from_merge_results(merge_results)
        if conflicts:
            comment = "\n".join([
                "Found %s conflicts in attempt to merge %s:%s to %s for %s." % \
                    (len(conflicts), startrev, endrev, self.version,
                     self.requestor),
                "Files in conflict:",
                "{{{",
                "\n".join(conflicts),
                "}}}",
                "A rebranch will be needed before this can be merged.",
            ])
            results['mergebotstate'] = 'conflicts'
            success = False
        else:
            # The merge worked, so commit the changes.
            commitmessage = "\n".join([
                "Ticket #%s: %s" % (self.ticket, self.summary),
                "    Merge of %s:%s to %s for %s." % (startrev, endrev,
                    self.version, self.requestor),
            ])
            committedrev = SvnOps.commit(checkoutdir, commitmessage, logfile)
            # Sed message and endstatus
            if committedrev == None:
                # Apparently nothing to commit.
                comment = "\n".join([
                    "Merged %s:%s to %s for %s." % (startrev, endrev,
                        self.version, self.requestor),
                    "No changes to commit.",
                ])
                results['mergebotstate'] = 'merged'
                success = True
            elif committedrev >= 0:
                # The commit succeeded.
                comment = "\n".join([
                    "Merged %s:%s to %s for %s." % (startrev, endrev,
                        self.version, self.requestor),
                    "Changeset [%s]. [source:%s/%s@%s]" % (committedrev,
                        self.component, self.version_subdir(), committedrev),
                ])
                results['mergebotstate'] = 'merged'
                success = True
            else:
                # The commit for the merge failed.
                comment = \
                    "Commit failed in attempt to merge %s:%s to %s for %s." \
                    % (startrev, endrev, self.version, self.requestor)
                #results['mergebotstate'] = 'mergefailed'
                success = False

        # Clean up the work area
        if os.path.exists(checkoutdir):
            shutil.rmtree(checkoutdir)

        return results, comment, success

# vim:foldcolumn=4 foldmethod=indent
# vim:tabstop=4 shiftwidth=4 softtabstop=4 expandtab
