#!/usr/bin/env python
"""Module for creating new branches for tickets"""

import time

from mergebot import SvnOps
from mergebot.Actor import Actor

class BranchActor(Actor):
    """This class handles creating a new branch for a ticket."""
    def execute(self):
        """Create the branch for the given ticket.
        """
        results = {}
        # Setup logging
        logfile = self.logfilename()
        open(logfile, "a").write("%s: branching ticket %s\n" % (time.asctime(),
            self.ticket))

        # Make sure the various urls we require do exist
        problems = self.check_required_directories()
        if problems:
            return results, problems, False

        commit_header = 'Ticket #%s: %s' % (self.ticket, self.summary)

        # Delete the branch if it already exists.  This can happen if the branch
        # was merged, but we're still working on it.
        if SvnOps.does_url_exist(self.branch_local_url()):
            # This branch already exists.
            commit_message = "\n".join([commit_header,
                "    Delete old branch",
            ])
            new_rev = SvnOps.delete_branch(self.branch_local_url(),
                                           commit_message, logfile)
            if new_rev == -1:
                results['mergebotstate'] = 'branchfailed'
                comment = 'Deleting the existing branch failed.'
                return results, comment, False

        # Do the branch creationg
        commit_message = "\n".join([commit_header,
            "    Create branch from %s for %s." % (self.version,
                                                   self.requestor),
        ])
        retval = SvnOps.create_branch(self.baseline_local_url(),
            self.branch_local_url(), commit_message, logfile)
        if retval:
            # Failed for some reason.
            results['mergebotstate'] = 'branchfailed'
            comment = 'Failed to create branch.'
            return results, comment, False

        results['mergebotstate'] = 'branched'
        comment = '\n'.join([
            'Created branch from %s for %s.' % (self.version, self.requestor),
            '',
            'Browse branch [source:%s/branches/ticket-%s source code] and ' \
                '[log:%s/branches/ticket-%s commit log].' % 
                (self.component, self.ticket, self.component, self.ticket),
            '',
            'To checkout, run:',
            '{{{',
            'svn checkout %s %s-%s' % (self.branch_public_url(),
                                       self.component, self.ticket),
            '}}}',
        ])
        return results, comment, True

# vim:foldcolumn=4 foldmethod=indent
# vim:tabstop=4 shiftwidth=4 softtabstop=4 expandtab
