#!/usr/bin/python
import os
import time

from rgbkbd.core import Monochrome, KeyboardMode
from rgbkbd.color import Colors
from rgbkbd.animation import RandomAnimation


class SecretNotesMode(KeyboardMode):
    """A KeyboardMode that will unbind all keys so the OS does not see what is
    typed, but it saves what is typed to a file in the user's home directory
    called .secret-<unixepochtime>.  In other words, this gives a discrete
    note-taking application.
    """
    # A mapping to give more readable output for the note taking files
    key_maps = {
        "common": { # same shift or no shift
            "enter": "\n",
            "numenter": "\n",
            "space": " ",
            "tab": "\t",
            "bspace": "\b",
        },
        False: { # no shift
            "bslash": "\\",
            "colon": ";",
            "comma": ",",
            "dot": ".",
            "equal": "=",
            "grave": "`",
            "lbrace": "[",
            "minus": "-",
            "quote": "'",
            "rbrace": "]",
            "slash": "/",
        },
        True: { # shift
            "bslash": "|",
            "colon": ":",
            "comma": "<",
            "dot": ">",
            "equal": "+",
            "grave": "~",
            "lbrace": "{",
            "minus": "_",
            "quote": "\"",
            "rbrace": "}",
            "slash": "?",
        },
    }
    for c in "abcdefghijklmnopqrstuvwxyz":
        key_maps[False][c] = c
        key_maps[True][c] = c.upper()
    for n, c in zip("1234567890", "!@#$%^&*()"):
        key_maps[False][n] = n
        key_maps[True][n] = c

    def __init__(self, manager, keyboard):
        super(SecretNotesMode, self).__init__(manager, keyboard,
            static_lighting=Monochrome(keyboard, color=Colors.BLACK),
            animations=[
                RandomAnimation(foreground=Colors.GREEN, background=Colors.BLACK, keyboard=keyboard),
            ])
        self.out = None
        self.state = {
            "shift": False,
        }

    def start(self, previous_mode=None):
        """Initialize the keyboard mode"""
        super(SecretNotesMode, self).start()
        # disable all keys so the OS doesn't see the keypresses
        self.keyboard.unbind()
        filename = os.path.expanduser("~/.secret-%s" % int(time.time()))
        self.out = open(filename, 'a')

    def event(self, key, state):
        """Handle key events"""
        if key in ["lshift", "rshift"]:
            self.state["shift"] = (state == '+')
        elif state == '+':
            # Translate key presses into characters to output.
            try:
                value = self.key_maps["common"][key]
            except KeyError:
                value  = self.key_maps[self.state["shift"]].get(key, "<%s>" % key)
            self.out.write(value)
            self.out.flush()
